using System;
using System.Windows.Forms;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using gov.va.med.vbecs;
using gov.va.med.vbecs.Common;
using System.Text.RegularExpressions;

namespace gov.va.med.vbecs.GUI.controls
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/10/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///User Control designed to allow users to enter an expiration date via barcode, manual entry or from date picker
	///</summary>

	#endregion


	public class ExpirationDateComboBox : GUI.controls.BaseControl
	{
		#region Variables

		private bool barcode = false;

		/// <summary>
		/// ISBTScan object
		/// </summary>
		public Common.ISBTScan isbtScan = Common.ISBTScan.Unknown;

		private bool _CorrectBarcode = true;
		private Regex _codabarExpDate;
		private Regex _isbtExpDate;
		
		#endregion

		#region Events/Event Handlers
		/// <summary>
		/// Public event to notify when Barcode property has changed value
		/// </summary>
		public event EventHandler BarcodeChanged;

		/// <summary>
		/// Raises event if a barcode is scanned into this text box that is not an expiration date.
		/// </summary>
		public event EventHandler OtherIsbtFieldScanned;

		/// <summary>
		/// Raises barcode indicator changed event
		/// </summary>
		protected virtual void OnBarcodeChanged()
		{
			this.OnUserActivity();
			if(BarcodeChanged != null)
				BarcodeChanged(this, EventArgs.Empty);
		}

		private BarcodeType barcodeType = BarcodeType.None;

		/// <summary>
		/// Public event to notify when BarcodeType property has changed value
		/// </summary>
		public event EventHandler BarcodeTypeChanged;

		/// <summary>
		/// Raises barcode type changed event
		/// </summary>
		protected virtual void OnBarcodeTypeChanged()
		{
			this.OnUserActivity();
			if(BarcodeTypeChanged != null)
				BarcodeTypeChanged(this, EventArgs.Empty);
		}

		private string iSBTBarcodeDate = string.Empty;

		/// <summary>
		/// Stores the ISBT string in cases where the wrong field is scanned.
		/// </summary>
		public string otherIsbtString;

		/// <summary>
		/// Public event to notify when IsbtBarcodeDate property has changed
		/// </summary>
		public event EventHandler IsbtBarcodeDateChanged;

		/// <summary>
		/// Raises event if a barcode is scanned into this text box
		/// that is not a Expiration date.
		/// </summary>
		protected virtual void OnOtherIsbtFieldScanned()
		{
			this.OnUserActivity();
			if(OtherIsbtFieldScanned != null)
				OtherIsbtFieldScanned(this, EventArgs.Empty);
		}

		/// <summary>
		/// Raises ISBT barcode date changed event
		/// </summary>
		protected virtual void OnIsbtBarcodeDateChanged()
		{
			this.OnUserActivity();
			if(IsbtBarcodeDateChanged != null)
				IsbtBarcodeDateChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Public event to notify when the Value property has changed
		/// </summary>
		public event EventHandler ValueChanged;

		/// <summary>
		/// Raises value changed event
		/// </summary>
		protected virtual void OnValueChanged()
		{
			this.OnUserActivity();
			if(ValueChanged != null)
				ValueChanged(this, EventArgs.Empty);
		}
		#endregion

		#region Form controls

		private System.Windows.Forms.DateTimePicker dtpDate;
		private System.Windows.Forms.TextBox txtDate;
		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		#endregion

		#region Constructors/Dispose
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// ExpirationDateComboBox user control constructor.
		/// </summary>
		public ExpirationDateComboBox()
		{
			InitializeComponent();
			try
			{
				SetComponentProperties();
			}
			catch
			{
				//try catch block used for design time, otherwise control cannot be used
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		/// <summary>
		/// ExpirationDateComboBox user control constructor. Adds control to container
		/// </summary>
		public ExpirationDateComboBox(System.ComponentModel.IContainer container)
		{
			container.Add(this);
			InitializeComponent();
			try
			{
				SetComponentProperties();
			}
			catch
			{
				//try catch block used for design time, otherwise control cannot be used
			}
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}
		#endregion

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.dtpDate = new System.Windows.Forms.DateTimePicker();
			this.txtDate = new System.Windows.Forms.TextBox();
			this.SuspendLayout();
			// 
			// dtpDate
			// 
			this.dtpDate.Anchor = (((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right);
			this.dtpDate.CustomFormat = "";
			this.dtpDate.Format = System.Windows.Forms.DateTimePickerFormat.Custom;
			this.dtpDate.Name = "dtpDate";
			this.dtpDate.TabIndex = 0;
			this.dtpDate.TabStop = false;
			this.dtpDate.CloseUp += new System.EventHandler(this.dtpDate_CloseUp);
			// 
			// txtDate
			// 
			this.txtDate.Anchor = (((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right);
			this.txtDate.BorderStyle = System.Windows.Forms.BorderStyle.None;
			this.txtDate.Location = new System.Drawing.Point(3, 3);
			this.txtDate.Name = "txtDate";
			this.txtDate.Size = new System.Drawing.Size(178, 13);
			this.txtDate.TabIndex = 1;
			this.txtDate.Text = "";
			this.txtDate.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.OnKeyPress);
			// 
			// ExpirationDateComboBox
			// 
			this.Controls.AddRange(new System.Windows.Forms.Control[] {
																		  this.txtDate,
																		  this.dtpDate});
			this.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.Name = "ExpirationDateComboBox";
			this.Size = new System.Drawing.Size(200, 20);
			this.Leave += new System.EventHandler(this.ExpirationDateComboBox_Leave);
			this.ResumeLayout(false);

		}
		#endregion

		#region Properties
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		///<summary>
		///Gets or sets expiration date value.
		///</summary>
		[
		Category("Behavior"),
		Description("Value"),
		Bindable(true),
		Browsable(true),
		]
		public DateTime Value
		{
			get
			{
				return this.dtpDate.Value;
			}
			set
			{
				if (value < this.dtpDate.MinDate)
				{
					value = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
				}
				this.dtpDate.Value = value;
				DisplayDate();
				this.OnValueChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		///<summary>
		///Gets or sets the minimum expiration date selectable
		///</summary>
		[
		Category("Behavior"),
		Description("Minimum Date"),
		Bindable(true),
		Browsable(true),
		]
		public DateTime MinDate
		{
			get
			{
				return this.dtpDate.MinDate;
			}
			set
			{
				this.dtpDate.MinDate = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		/// <summary>
		/// Gets or sets the maximum expiration date selectable
		/// </summary>
		[
		Category("Behavior"),
		Description("Maximum Date"),
		Bindable(true),
		Browsable(true),
		]
		public DateTime MaxDate
		{
			get
			{
				return this.dtpDate.MaxDate;
			}
			set
			{
				this.dtpDate.MaxDate = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		/// <summary>
		/// Gets or sets value to indicate if entry came from a barcode scan
		/// </summary>
		[
		Category("Behavior"),
		Description("Barcode"),
		Bindable(true),
		Browsable(true),
		]
		public bool Barcode
		{
			get
			{
				return this.barcode;
			}
			set
			{
				this.barcode = value;
				OnBarcodeChanged();
			}
		}

		/// <summary>
		/// Get CorrectBarcode
		/// </summary>
		public bool CorrectBarcode
		{
			get
			{
				return this._CorrectBarcode;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		/// <summary>
		/// Gets or sets value indicating type of barcode entry
		/// </summary>
		[
		Category("Behavior"),
		Description("Barcode Type"),
		Bindable(true),
		Browsable(true),
		]
		public BarcodeType BarcodeType
		{
			get
			{
				return barcodeType; 
			}
			set
			{
				barcodeType = value;
				OnBarcodeTypeChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		///<summary>
		///Gets or sets expiration date as julian date and time format(cyyjjjhhmm)
		///</summary>
		public string IsbtBarcodeDate
		{
			get
			{
				return iSBTBarcodeDate;
			}
			set
			{
				iSBTBarcodeDate = value;
				OnIsbtBarcodeDateChanged();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/10/2002</CreationDate>
		/// <summary>
		/// Gets value to indicate if a valid expiration date has been entered
		/// </summary>
		public bool IsValid
		{
			get
			{
				if (this.txtDate.Text.Length > 0)
				{
					try
					{
						DateTime _dt = System.Convert.ToDateTime(txtDate.Text);
						if ((this.dtpDate.MinDate<_dt) && (this.dtpDate.MaxDate>_dt))
						{
							return true;
						}
						else
						{
							return false;
						}
					}
					catch
					{
						return false;
					}
				}
				else
				{
					return false;
				}
			}
		}


		/// <summary>
		/// Returns the text that's showing to the user -- allows for checking if date was enetered
		/// </summary>
		public string VisibleText
		{
			get
			{
				return(this.txtDate.Text);
			}
		}


		#endregion

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Override for KeyPress event. Checks for ~ character which denotes a barcode was scanned.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void OnKeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			this.OnUserActivity();

			base.OnKeyPress(e);
			if (e.KeyChar == '~')
			{
				this.Barcode = true;
				this.Text = this.Text.Replace("~","");
				e.Handled = true;
				return;
			}
			else if (e.KeyChar == (char)13)
			{
				e.Handled = true;
				return;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Leave event for user control. Determines expiration date from various valid entry formats.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ExpirationDateComboBox_Leave(object sender, System.EventArgs e)
		{
			if (this.txtDate.Text == string.Empty) return;

			isbtScan = Common.Utility.GetISBTFromString(this.txtDate.Text);

			this._CorrectBarcode = true;
			//
			this.IsbtBarcodeDate = string.Empty;
			string _txt = this.txtDate.Text;

			//BR_1.23
			if (_isbtExpDate.IsMatch(_txt))
			{
				this.BarcodeType = BarcodeType.ISBT128;
				_txt = _txt.Replace("&>","");
				this.IsbtBarcodeDate = _txt;
				// Julian Date!
				try
				{
					string _julian = "01/01/";
					byte _checkCentury = System.Convert.ToByte(_txt .Substring(0,1));
					if(_checkCentury == 9)
					{
						_julian = _julian + "1"+_txt.Substring(0,3);
					}
					else
					{
						_julian = _julian + "2"+_txt.Substring(0,3);
					}					
					double _julDays = System.Convert.ToDouble(_txt.Substring(3,3));
					this.dtpDate.Value = System.Convert.ToDateTime(_julian).AddDays(_julDays-1);
					if (_txt.Length == 10)
					{
						this.dtpDate.Value = this.dtpDate.Value.AddHours(Convert.ToDouble(_txt.Substring(6,2)));
						this.dtpDate.Value = this.dtpDate.Value.AddMinutes(Convert.ToDouble(_txt.Substring(8,2)));
					}
					else
					{
						this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
						this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
					}
				}
				catch
				{
					if (isbtScan != Common.ISBTScan.ExpirationDate && _txt.Trim().Length > 0)
					{
						this._CorrectBarcode = false;
					}
					InvalidDate();
					return;
				}
			}
			//BR_1.22
			else if (_codabarExpDate.IsMatch(_txt))
			{
				if (_txt.IndexOf("/") < 0)
				{
					//change format to mm/dd/yyyy
					_txt = _txt.Insert(2,"/");
					_txt = _txt.Insert(5,"/");
				}
				try
				{
					//check if valid date
					this.dtpDate.Value = System.Convert.ToDateTime(_txt).Date;
					this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
					this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
				}
				catch
				{
					InvalidDate();
					// If the user scans an ISBT field other that expiration date into this field.
					if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
					{
						this.otherIsbtString = _txt;
						this._CorrectBarcode = false;
						OnOtherIsbtFieldScanned();			
					}
					if (isbtScan != Common.ISBTScan.ExpirationDate && _txt.Trim().Length > 0)
					{
						this._CorrectBarcode = false;
					}
					return;
				}
				this.BarcodeType = BarcodeType.Codabar;
			}
			else
			{
				try
				{
					//check if valid date

					this.dtpDate.Value = System.Convert.ToDateTime(_txt).Date;

					if (System.Convert.ToDateTime(_txt).Hour > 0 || System.Convert.ToDateTime(_txt).Minute > 0)
					{
						this.dtpDate.Value = dtpDate.Value.AddHours(System.Convert.ToDateTime(_txt).Hour).AddMinutes(System.Convert.ToDateTime(_txt).Minute);
					}
					else
					{
						this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
						this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
					}
				}
				catch
				{
					InvalidDate();
					// If the user scans an ISBT field other that expiration date into this field.
					if (isbtScan != Common.ISBTScan.ExpirationDate && _txt.Trim().Length > 0)
					{
						this._CorrectBarcode = false;
					}
					if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
					{
						this.otherIsbtString = _txt;
						OnOtherIsbtFieldScanned();			
					}
					return;
				}
				this.BarcodeType = BarcodeType.None;
			}
			DisplayDate();
			if (this.IsbtBarcodeDate.Length == 0)
			{
				this.IsbtBarcodeDate = GetIsbtBarcodeDate(this.txtDate.Text);
			}
			// If the user scans an ISBT field other that expiration date into this field.
			if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
			{
				this.otherIsbtString = _txt;
				this._CorrectBarcode = false;
				OnOtherIsbtFieldScanned();			
			}
		}

		/// <summary>
		/// Set the date in the control and format it properly
		/// </summary>
		/// <param name="expirationDate"></param>
		public void SetControl(String expirationDate)
		{
			this.txtDate.Text = expirationDate;

			isbtScan = Common.Utility.GetISBTFromString(this.txtDate.Text);
			//
			this.IsbtBarcodeDate = string.Empty;
			string _txt = this.txtDate.Text;
			//BR_1.23
			if (_isbtExpDate.IsMatch(_txt))
			{
				this.BarcodeType = BarcodeType.ISBT128;
				_txt = _txt.Replace("&>","");
				this.IsbtBarcodeDate = _txt;
				// Julian Date!
				try
				{
					string _julian = "01/01/";
					byte _checkCentury = System.Convert.ToByte(_txt .Substring(0,1));
					if(_checkCentury == 9)
					{
						_julian = _julian + "1"+_txt.Substring(0,3);
					}
					else
					{
						_julian = _julian + "2"+_txt.Substring(0,3);
					}					
					double _julDays = System.Convert.ToDouble(_txt.Substring(3,3));
					this.dtpDate.Value = System.Convert.ToDateTime(_julian).AddDays(_julDays-1);
					if (_txt.Length == 10)
					{
						this.dtpDate.Value = this.dtpDate.Value.AddHours(Convert.ToDouble(_txt.Substring(6,2)));
						this.dtpDate.Value = this.dtpDate.Value.AddMinutes(Convert.ToDouble(_txt.Substring(8,2)));
					}
					else
					{
						this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
						this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
					}
				}
				catch
				{
					InvalidDate();
					return;
				}
			}
				//BR_1.22
			else if (_codabarExpDate.IsMatch(_txt))
			{
				if (_txt.IndexOf("/") < 0)
				{
					//change format to mm/dd/yyyy
					_txt = _txt.Insert(2,"/");
					_txt = _txt.Insert(5,"/");
				}
				try
				{
					//check if valid date
					this.dtpDate.Value = System.Convert.ToDateTime(_txt).Date;
					this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
					this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
				}
				catch
				{
					InvalidDate();
					// If the user scans an ISBT field other that expiration date into this field.
					if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
					{
						this.otherIsbtString = _txt;
						OnOtherIsbtFieldScanned();			
					}
					return;
				}
				this.BarcodeType = BarcodeType.Codabar;
			}
			else
			{
				try
				{
					//check if valid date
					this.dtpDate.Value = System.Convert.ToDateTime(_txt).Date;
					this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
					this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
				}
				catch
				{
					InvalidDate();
					// If the user scans an ISBT field other that expiration date into this field.
					if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
					{
						this.otherIsbtString = _txt;
						OnOtherIsbtFieldScanned();			
					}
					return;
				}
				this.BarcodeType = BarcodeType.None;
			}
			DisplayDate();
			if (this.IsbtBarcodeDate.Length == 0)
			{
				this.IsbtBarcodeDate = GetIsbtBarcodeDate(this.txtDate.Text);
			}
			// If the user scans an ISBT field other that expiration date into this field.
			if (isbtScan != Common.ISBTScan.ExpirationDate && isbtScan != Common.ISBTScan.Unknown)
			{
				this.otherIsbtString = _txt;
				OnOtherIsbtFieldScanned();			
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Event fired when date time picker closes. Defaults the time to 11:59 PM instead of 12:00 AM
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void dtpDate_CloseUp(object sender, System.EventArgs e)
		{
			this.dtpDate.Value = this.dtpDate.Value.Date;
			this.dtpDate.Value = this.dtpDate.Value.AddHours(23);
			this.dtpDate.Value = this.dtpDate.Value.AddMinutes(59);
			DisplayDate();
			OnValueChanged();
			this.txtDate.Focus();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to display the expiration date. If the date chosen is the same as the MinDate, then no date is displayed 
		/// </summary>
		private void DisplayDate()
		{
			this.OnUserActivity();
			if (dtpDate.Value == this.MinDate)
			{
				this.txtDate.Text = string.Empty;
			}
			else
			{
				this.txtDate.Text = Common.VBECSDateTime.FormatDateTimeString(this.dtpDate.Value);
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to clear text in control
		/// </summary>
		public void ClearDateText()
		{
			this.txtDate.Text = string.Empty;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to reset properties when an expiration date has been found to be invalid
		/// </summary>
		private void InvalidDate()
		{
			this.BarcodeType = BarcodeType.None;
			this.dtpDate.Value = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			DisplayDate();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private function to return an ISBT format julian date (cyyjjjhhmm) from a non barcode date entry
		/// </summary>
		/// <param name="dateString"></param>
		/// <returns></returns>
		private string GetIsbtBarcodeDate(string dateString)
		{
			DateTime _date = Convert.ToDateTime(dateString).Date;
			string _year = _date.Year.ToString().Substring(1,3);
			System.Globalization.JulianCalendar _julian = new System.Globalization.JulianCalendar();
			_julian.GetDayOfYear(_date.Date);
			string _days = "000"+_julian.GetDayOfYear(_date.Date).ToString();
			_days = _days.Substring(_days.Length-3,3);
			return _year+_days+"2359";
		}

		private void SetComponentProperties()
		{
			//set custom date and time format
			this.dtpDate.CustomFormat = Common.VBECSDateTime.DateTimeFormat;

			//set current date value based on the division time zone
			this.dtpDate.Value = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._codabarExpDate = Common.RegularExpressions.CodabarExpirationDate();
			this._isbtExpDate = Common.RegularExpressions.ISBT128ExpirationDate();

			this.dtpDate.TabStop = false;
		}
	}
}
